<?php

namespace App\Http\Controllers\Website;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\PropertyRequirement;
use Illuminate\Support\Facades\Gate;

class PropertyRequirementController extends Controller
{
    /**
     * Store a new property requirement
     */
    public function store(Request $request)
    {
        $request->validate([
            'user_type' => 'required|in:seeker,landlord,broker',
            'price_range' => 'required|in:more_than_2m,1m_to_2m,500k_to_1m,less_than_500k',
            'city_id' => 'required|exists:cities,id',
            'category_id' => 'required|exists:categories,id',
            'full_name' => 'required|string|max:255',
            'terms_accepted' => 'required|boolean',
             'phone' => 'required|string|max:20',
             'email' => 'required|email|max:255',
        ]);

        $requirement = PropertyRequirement::create(array_merge($request->all(), ['status' => 1]));

        return response()->json([
            'status' => true,
            'message' => 'Property requirement submitted successfully',
            'data' => $requirement->load('city','category')
        ]);
    }

    /**
     * List all property requirements (optionally filter by status)
     */
    public function index(Request $request)
    {
     Gate::authorize('view-property-requirments');

        $query = PropertyRequirement::with('city', 'category');

        if ($request->has('status')) {
            $query->where('status', $request->status);
        }

        $requirements = $query->get();

        return response()->json([
            'status' => true,
            'data' => $requirements
        ]);
    }

    /**
     * Update status (0 or 1)
     */
    public function updateStatus(Request $request, $id)
    {
                Gate::authorize('update-property-requirments');

        $request->validate([
            'status' => 'required|in:0,1',
        ]);

        $requirement = PropertyRequirement::findOrFail($id);
        $requirement->status = $request->status;
        $requirement->save();

        return response()->json([
            'status' => true,
            'message' => 'Status updated successfully',
            'data' => $requirement
        ]);
    }
    
        public function show($id)
    {
                Gate::authorize('show-property-requirments');

        $requirement = PropertyRequirement::with('city', 'category')->findOrFail($id);
    
        return response()->json([
            'status' => true,
            'data' => $requirement
        ]);
    }
    public function destroy($id)
{
    Gate::authorize('delete-property-requirments');

    $requirement = PropertyRequirement::findOrFail($id);
    $requirement->delete();

    return response()->json([
        'status' => true,
        'message' => 'Property requirement deleted successfully'
    ]);
}

}
