<?php

namespace App\Http\Controllers\Website;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;

class AuthController extends Controller
{
    // REGISTER
    public function register(Request $request)
    {
        $request->validate([
            'name' => 'required',
            'phone' => 'required|unique:users',
            'email' => 'nullable|email|unique:users',
            'password' => 'required|min:6'
        ]);

        $user = User::create([
            'name_ar' => $request->name,
            'name_en' => $request->name,
            'email' => $request->email,
            'phone' => $request->phone,
            'password' => Hash::make($request->password),
        ]);


        return response()->json([
            'status' => true,
            'user' => [
                'id' => $user->id,
                'name' => $user->name_ar,
                'email' => $user->email,
                'phone' => $user->phone,
            ]
        ]);
    }

    // LOGIN
    public function login(Request $request)
    {
        $request->validate([
            'phone' => 'required',
            'password' => 'required'
        ]);

        $user = User::where('phone', $request->phone)->first();

        if (!$user || !Hash::check($request->password, $user->password)) {
            return response()->json([
                'status' => false,
                'message' => 'Invalid credentials'
            ], 401);
        }

        $token = $user->createToken('website-token')->plainTextToken;

        return response()->json([
            'status' => true,
            'token' => $token,
            'user' => [
                'id' => $user->id,
                'name' => $user->name_ar,
                'email' => $user->email,
                'phone' => $user->phone,
            ]
        ]);
    }


    // LOGOUT
    public function logout(Request $request)
    {
        $request->user()->currentAccessToken()->delete();

        return response()->json([
            'status' => true,
            'message' => 'Logged out successfully'
        ]);
    }


    public function updateProfile(Request $request)
    {
        $user = Auth::user();

        $request->validate([
            'name' => 'sometimes|required|string|max:255',
            'phone' => 'sometimes|required|unique:users,phone,' . $user->id,
            'email' => 'nullable|email|unique:users,email,' . $user->id,
        ]);

        $data = [];

        if ($request->has('name')) {
            $data['name_ar'] = $request->name;
            $data['name_en'] = $request->name;
        }

        if ($request->has('phone')) {
            $data['phone'] = $request->phone;
        }

        if ($request->has('email')) {
            $data['email'] = $request->email;
        }

        $user->update($data);

        return response()->json([
            'status' => true,
            'message' => 'Profile updated successfully',
            'user' => [
                'id' => $user->id,
                'name' => $user->name_ar,
                'email' => $user->email,
                'phone' => $user->phone,
            ]
        ]);
    }
    public function updatePassword(Request $request)
    {
        $user = Auth::user();

        $request->validate([
            'current_password' => 'required',
            'password' => 'required|min:6|confirmed'
        ]);

        if (!Hash::check($request->current_password, $user->password)) {
            return response()->json([
                'status' => false,
                'message' => 'Current password is incorrect'
            ], 422);
        }

        $user->password = Hash::make($request->password);
        $user->save();

        $user->tokens()->delete();

        return response()->json([
            'status' => true,
            'message' => 'Password updated successfully. All sessions logged out.'
        ]);
    }
    
    
           public function authUser(Request $request)
    {
        $user = auth()->user();
    
        unset(
            $user->status,
            $user->home_status,
            $user->sort,
            $user->type,
            $user->is_best
        );
    
        return response()->json([
            'status' => true,
            'data' => $user
        ]);
    }


}
