<?php

namespace App\Actions\V1\Project;

use App\Dto\V1\Project\ProjectDto;
use App\Exceptions\LogicalException;
use App\Models\ProjectImage;
use App\Repositories\V1\Projects\ProjectRepositoryInterface;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;

readonly class UpdateProjectAction
{
    public function __construct(
        private ProjectRepositoryInterface $projectRepository
    ) {}

    /**
     * @throws LogicalException
     */
    public function execute(int $projectId, ProjectDto $dto, int $userId): void
    {
        $project = $this->projectRepository->first($projectId)?->load('images');
        if (!$project || $project->user_id != $userId) {
            throw new ModelNotFoundException(__('exceptions.project_not_found'));
        }

        $imagesToRemove = ProjectImage::whereIn('id', $dto->removedImages ?? [])
            ->where('project_id', $projectId)
            ->get();
        $uploadFiles = array_values(array_filter(
            $dto->images ?? [],
            fn($file) => $file instanceof UploadedFile
        ));


        $finalCount = ($project->images->count() - $imagesToRemove->count()) + count($uploadFiles);
        if ($finalCount > config('general.project_image_limit')) {
            throw new LogicalException(__('exceptions.images_max_has_been_exceeded'));
        }

        $storedPaths = [];

        DB::beginTransaction();
        try {
            $this->projectRepository->update($projectId, $dto->projectUpdatePayload());

            foreach ($imagesToRemove as $image) {
                Storage::disk('public')->delete($image->image);
                $image->delete();
            }

            foreach ($uploadFiles as $file) {
                $path = $file->store("projects/{$projectId}", 'public');
                $storedPaths[] = $path;

                ProjectImage::create([
                    'project_id' => $projectId,
                    'image' => $path,
                ]);
            }

            DB::commit();
        } catch (\Throwable $e) {
            DB::rollBack();

            if (!empty($storedPaths)) {
                Storage::disk('public')->delete($storedPaths);
            }

            throw $e;
        }
    }
}
