<?php

namespace App\Models;

use App\Models\Traits\ProjectRelations;
use App\Enums\Project\OfferType;
use App\Enums\Project\Condition;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Casts\Attribute;
use Carbon\Carbon;

class Project extends Model
{
    use ProjectRelations;

    protected $guarded = [];
    protected $appends = [
        'main_image', 
        'thumbnail_image',
        'price_formatted',
        'handover_date_formatted',
        'offer_type_text',
        'condition_text',
        'facilities'
    ];
    
    protected $with = ['firstImage'];

    protected $casts = [
        'handover_date' => 'date',
        // 'has_pool' => 'boolean',
        // 'has_gym' => 'boolean',
        // 'has_parking' => 'boolean',
        // 'furnished' => 'boolean',
        'price' => 'decimal:2',
        'total_area' => 'decimal:2',
        'land_area' => 'decimal:2',
        'built_area' => 'decimal:2',
        'down_payment' => 'decimal:2',
        'installment_amount' => 'decimal:2',
        'offer_type' => OfferType::class,
        'condition' => Condition::class,
    ];

    // Multi-language Accessors
    protected function name(): Attribute
    {
        return Attribute::make(
            get: fn () => $this->{"name_" . app()->getLocale()} 
                ?? $this->{"name_" . (app()->getLocale() === 'ar' ? 'en' : 'ar')} 
                ?? ''
        );
    }

    protected function area(): Attribute
    {
        return Attribute::make(
            get: fn () => $this->{"area_" . app()->getLocale()} 
                ?? $this->{"area_" . (app()->getLocale() === 'ar' ? 'en' : 'ar')} 
                ?? ''
        );
    }

    protected function description(): Attribute
    {
        return Attribute::make(
            get: fn () => $this->{"description_" . app()->getLocale()} 
                ?? $this->{"description_" . (app()->getLocale() === 'ar' ? 'en' : 'ar')} 
                ?? ''
        );
    }

    // Real Estate Accessors
    protected function priceFormatted(): Attribute
    {
        return Attribute::make(
            get: fn () => $this->price 
                ? number_format($this->price, 0) . ' ' . strtoupper($this->price_currency ?? 'SAR')
                : ' undefined'
        );
    }

    protected function handoverDateFormatted(): Attribute
    {
        return Attribute::make(
            get: fn () => $this->handover_date 
                ? $this->handover_date->format('Y-m-d') 
                : 'undefined '
        );
    }

    protected function offerTypeText(): Attribute
    {
        return Attribute::make(
            get: fn () => $this->offer_type?->label() ?? 'undefined'
        );
    }

    protected function conditionText(): Attribute
    {
        return Attribute::make(
            get: fn () => $this->condition?->label() ?? 'undefined '
        );
    }

    protected function facilities(): Attribute
    {
        return Attribute::make(
            get: fn () => collect()
                ->when($this->has_pool, fn($c) => $c->push('pool'))
                ->when($this->has_gym, fn($c) => $c->push('gym'))
                ->when($this->has_parking, fn($c) => $c->push('parking'))
                ->when($this->furnished, fn($c) => $c->push('furnished'))
                ->values()
                ->all()
        );
    }

    // Image Accessors
    protected function mainImage(): Attribute
    {
        return Attribute::make(
            get: fn () => $this->firstImage?->image
        );
    }

    protected function thumbnailImage(): Attribute
    {
        return Attribute::make(
            get: fn () => $this->firstImage?->image
        );
    }

    // Scopes
    public function scopeRealEstate($query)
    {
        return $query->whereNotNull('price');
    }

    public function scopeForSale($query)
    {
        return $query->where('offer_type', OfferType::SALE->value);
    }

    public function scopeForRent($query)
    {
        return $query->where('offer_type', OfferType::RENT->value);
    }

    public function scopeNewProjects($query)
    {
        return $query->where('condition', Condition::NEW->value);
    }

    // Relations
    public function designType()
    {
        return $this->belongsTo(DesignType::class);
    }

    public function projectLikes()
    {
        return $this->hasMany(ProjectLike::class, 'project_id', 'id');
    }

    public function likes()
    {
        return $this->hasMany(ProjectLike::class);
    }

    public function country()
    {
        return $this->belongsTo(Country::class);
    }

    public function subCategory()
    {
        return $this->belongsTo(SubCategory::class, 'sub_category_id');
    }

    // Helper methods
    public function getLikesCountAttribute(): int
    {
        return $this->likes()->count();
    }

    public function isLikedBy($userId): bool
    {
        return $this->likes()->where('user_id', $userId)->exists();
    }
}
