<?php

namespace App\Http\Requests\V1\Projects;

use App\Enums\Project\ProjectTypeEnum;
use App\Enums\Project\OfferType;
use App\Enums\Project\Condition;
use App\Enums\StatusEnum;
use App\Http\Requests\BaseFormRequest;
use Illuminate\Validation\Rule;
use Illuminate\Validation\Rules\Enum;

class UpdateProjectRequest extends BaseFormRequest
{
    public function authorize(): bool
    {
        return true;
    }

    public function rules(): array
    {
        return [

            // =======================
            // Basic Fields
            // =======================

            'name_en' => ['nullable', 'string', 'max:255'],
            'name_ar' => ['nullable', 'string', 'max:255'],
            'description_en' => ['nullable', 'string'],
            'description_ar' => ['nullable', 'string'],
            'area_en' => ['nullable', 'string'],
            'area_ar' => ['nullable', 'string'],
            'sort' => ['nullable', 'integer'],

            // =======================
            // Project Type & Category
            // =======================

            'type' => ['nullable', 'numeric', new Enum(ProjectTypeEnum::class)],
            'category_id' => ['nullable', 'numeric', Rule::exists('categories', 'id')],
            'sub_category_id' => ['nullable', 'integer', Rule::exists('sub_categories', 'id')],

            // =======================
            // Location
            // =======================

            'country_id' => ['nullable', 'integer', Rule::exists('countries', 'id')],

            'state_id' => [
                'nullable',
                'numeric',
                Rule::exists('states', 'id')
                    ->where('status', StatusEnum::ACTIVE->value)
            ],

            'city_id' => [
                'nullable',
                'numeric',
                Rule::exists('cities', 'id')
                    ->where('state_id', $this->state_id)
                    ->where('status', StatusEnum::ACTIVE->value)
            ],

            // =======================
            // External Links
            // =======================

            'tiktok_link' => ['nullable', 'string', 'url', 'max:255'],

            // =======================
            // Images
            // =======================

            'images' => ['nullable', 'array', 'max:' . config('general.project_image_limit')],
            'images.*' => ['file', 'image', 'max:5120'],

            'removed_images' => ['nullable', 'array'],

            // =======================
            // Rooms & Numbers
            // =======================

            'number_of_bathrooms' => ['nullable', 'integer', 'min:0'],
            'number_of_rooms' => ['nullable', 'integer', 'min:0'],
            'number_of_beds' => ['nullable', 'integer', 'min:0'],

        ] + $this->projectRules() + $this->realEstateRules();
    }

    /**
     * Project Related Rules
     */
    private function projectRules(): array
    {
        return [

            'project_type_id' => [
                'nullable',
                'numeric',
                Rule::exists('project_types', 'id')
                    ->where('status', StatusEnum::ACTIVE->value)
            ],

            'design_type_id' => ['nullable', 'numeric', Rule::exists('design_types', 'id')],

            'budget_id' => ['nullable', 'numeric', Rule::exists('budgets', 'id')],
        ];
    }

    /**
     * Real Estate Rules
     */
    private function realEstateRules(): array
    {
        return [

            // ===== Price =====

            'price' => ['nullable', 'numeric', 'min:0'],
            'price_currency' => ['nullable', 'in:SAR,USD,EUR,AED'],

            // ===== Areas =====

            'total_area' => ['nullable', 'numeric', 'min:0'],
            'land_area' => ['nullable', 'numeric', 'min:0', 'lte:total_area'],
            'built_area' => ['nullable', 'numeric', 'min:0', 'lte:total_area'],

            // ===== Counts =====

            'total_floors' => ['nullable', 'integer', 'min:0', 'max:100'],
            'total_units' => ['nullable', 'integer', 'min:0'],

            // ===== Date =====

            'handover_date' => [
                'nullable',
                'date_format:Y-m-d',
                'after:today',
            ],


            // ===== Offer & Condition =====

            'offer_type' => ['nullable', new Enum(OfferType::class)],
            'condition' => ['nullable', new Enum(Condition::class)],

            // ===== Features =====

            'has_pool' => ['boolean'],
            'has_gym' => ['boolean'],
            'has_parking' => ['boolean'],
            'furnished' => ['boolean'],

            // ===== Installments =====

            'down_payment' => ['nullable', 'numeric', 'min:0', 'lte:price'],
            'installments_count' => ['nullable', 'integer', 'min:1', 'max:120'],
            'installment_amount' => ['nullable', 'numeric', 'min:0'],
        ];
    }
    
    protected function prepareForValidation(): void
{
    if ($this->filled('handover_date')) {
        $date = $this->handover_date;

        if (preg_match('/^\d{2}-\d{2}-\d{4}$/', $date)) {
            $this->merge([
                'handover_date' => \Carbon\Carbon::createFromFormat('d-m-Y', $date)->format('Y-m-d'),
            ]);
        }
    }
}


   
}
