<?php

namespace App\Http\Requests\V1\Projects;

use App\Enums\Project\ProjectTypeEnum;
use App\Enums\Project\OfferType;
use App\Enums\Project\Condition;
use App\Enums\StatusEnum;
use App\Http\Requests\BaseFormRequest;
use Illuminate\Validation\Rule;
use Illuminate\Validation\Rules\Enum;

class StoreProjectRequest extends BaseFormRequest
{
    public function authorize(): bool
    {
        return true; 
    }

    public function rules(): array
    {
        return [
            // الحقول الأساسية
            'name_en' => ['required', 'string', 'max:255'],
            'name_ar' => ['required', 'string', 'max:255'],
            'description_en' => ['required', 'string'],
            'description_ar' => ['required', 'string'],
            'area_en' => ['required', 'string'],
            'area_ar' => ['required', 'string'],
            'sort' => ['required', 'int'],

            // نوع المشروع والفئات
            'type' => ['required', 'numeric', new Enum(ProjectTypeEnum::class)],
            'category_id' => ['required', 'numeric', Rule::exists('categories', 'id')],
            'sub_category_id' => ['nullable', 'integer', Rule::exists('sub_categories', 'id')],

            // الموقع
            'country_id' => ['nullable', 'integer', Rule::exists('countries', 'id')],
            'state_id' => [
                'required',
                'numeric',
                Rule::exists('states', 'id')->where('status', StatusEnum::ACTIVE->value)
            ],
            'city_id' => [
                'required',
                'numeric',
                Rule::exists('cities', 'id')
                    ->where('state_id', $this->state_id)
                    ->where('status', StatusEnum::ACTIVE->value)
            ],

            // روابط خارجية
            'tiktok_link' => ['nullable', 'string', 'url', 'max:255'],

            // الصور
            'images' => ['required', 'array', 'max:' . config('general.project_image_limit')],
            'images.*' => ['file', 'image', 'max:5120'], 

            // 👇 حقول العقارات الجديدة
            'price' => ['required', 'numeric', 'min:0'],
            'price_currency' => ['required', 'in:SAR,USD,EUR,AED'],
            
            'total_area' => ['nullable', 'numeric', 'min:0'],
            'land_area' => ['nullable', 'numeric', 'min:0', 'lte:total_area'],
            'built_area' => ['nullable', 'numeric', 'min:0', 'lte:total_area'],
            
            'total_floors' => ['nullable', 'integer', 'min:0', 'max:100'],
            'total_units' => ['nullable', 'integer', 'min:0'],
            
            'handover_date' => ['nullable', 'date', 'after:'.now()->format('Y-m-d')],
            
            'offer_type' => ['required', new Enum(OfferType::class)],
            'condition' => ['required', new Enum(Condition::class)],
            
            'has_pool' => ['boolean'],
            'has_gym' => ['boolean'],
            'has_parking' => ['boolean'],
            'furnished' => ['boolean'],
            
            'down_payment' => ['nullable', 'numeric', 'min:0', 'lte:price'],
            'installments_count' => ['nullable', 'integer', 'min:1', 'max:120'],
            'installment_amount' => ['nullable', 'numeric', 'min:0'],
            'number_of_bathrooms' => ['nullable', 'integer', 'min:0'],
            'number_of_rooms' => ['nullable', 'integer', 'min:0'], 
            'number_of_beds' => ['nullable', 'integer', 'min:0'],

        ];
    }

    public function messages(): array
    {
        return [
            'price.required' => 'السعر مطلوب',
            'price_currency.in' => 'العملة يجب أن تكون SAR أو USD أو EUR أو AED',
            'land_area.lte' => 'مساحة الأرض لا يمكن أن تتجاوز المساحة الإجمالية',
            'built_area.lte' => 'مساحة البناء لا يمكن أن تتجاوز المساحة الإجمالية',
            'down_payment.lte' => 'المقدم لا يمكن أن يتجاوز السعر الكلي',
            'handover_date.after' => 'تاريخ التسليم يجب أن يكون في المستقبل',
        ];
    }
}
